{******************************************************************************
 *  bits.pp - basic bits operations
 *
 *
 *  Copyright (c) 2004 nucleOS Group [http://nucle-os.sourceforge.net/]
 *                                   [http://www.sf.net/projects/nucle-os]
 *                                   [http://www.saint-soft.de/nucleos/board/]
 *
 *  version 0.1 - 21/02/2004 - first release
 *
 *  written by
 *    Michael Gerh"auser (saberrider) [saberrider@users.sourceforge.net]
 *
 *  This unit implements some functions around bits
 *
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software Foundation
 * (version 2, June 1991)
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program; if not, write to:
 * Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330
 * Boston, MA 02111-1307 USA
 *
 ******************************************************************************}


unit bits;


interface


{* Procedures and functions defined in this file *}
function set_bit( index: byte; value: Pointer ): Pointer;
function unset_bit( index: byte; value: Pointer ): Pointer;
function is_bit_set( index: byte; value: Pointer ): Boolean;
function is_bit_not_set( index: byte; value: Pointer ): Boolean;
function scan_bits( value: Pointer ): byte;
function high( value: DWord ): Word;
function high( value: word ): byte;
function low( value: DWord ): Word;
function low( value: Word ): byte;

const
     bit_mask : array [0..31] of DWord = ($1, $2, $4, $8, $10, $20, $40, $80,
                                          $100, $200, $400, $800, $1000,
                                          $2000, $4000, $8000, $10000,
                                          $20000, $40000, $80000,
                                          $100000, $200000, $400000,
                                          $800000, $1000000, $2000000,
                                          $4000000, $8000000, $10000000,
                                          $20000000, $40000000, $80000000);

implementation

{******************************************************************************
 *    in following bit functions the bit index always starts at most right pos
 *    e.g.:
 *    index:   31 30 29 28 27 26        8  7  6  5  4  3  2  1  0
 *    value:   0  0  0  0  1  1  (...)  1  0  1  0  0  1  1  0  1
 *
 ******************************************************************************}


{******************************************************************************
 *   set_bit
 ******************************************************************************
 *  does:
 *   sets bit at index
 ******************************************************************************}
function set_bit( index: byte; value: Pointer ): Pointer; [public, alias:'SET_BIT'];
var
   tmp: Pointer;
begin
     tmp := Pointer(bit_mask[index]);

     asm
        mov eax, value
        mov ebx, tmp
        or eax, ebx
        mov tmp, eax
     end;

     set_bit := tmp;
end;

{******************************************************************************
 *   unset_bit
 ******************************************************************************
 *  does:
 *   clears bit at index
 ******************************************************************************}
function unset_bit( index: byte; value: Pointer ): Pointer; [public, alias:'UNSET_BIT'];
var
   tmp: Pointer;
begin
     tmp := Pointer(bit_mask[index]);

     asm
        mov eax, value
        mov ebx, tmp
        not eax
        or eax, ebx
        not eax
        mov tmp, eax
     end;

     unset_bit := tmp;
end;

{******************************************************************************
 *   scan_bits
 ******************************************************************************
 *  does:
 *   scans value from right to left bit by bit for unset bits
 *   any time an unset bit is found, counter is incremented
 *   so whenever more than one bit is cleared, scan_bits always returns
 *   the last unset bit with the highest index
 ******************************************************************************}
function scan_bits( value: Pointer ): byte; [public, alias:'SCAN_BITS'];
var
   tmp, i: byte;
begin
     tmp := 32;

     for i := 0 to 31 do
     begin
         if (is_bit_not_set(i, value)) then
            tmp := i;
     end;

     scan_bits := tmp;
end;

{******************************************************************************
 *   is_bit_set
 ******************************************************************************
 *  does:
 *   returns true if bit at given index is set
 ******************************************************************************}
function is_bit_set( index: byte; value: Pointer ): Boolean; [public, alias:'IS_BIT_SET'];
var
   tmp: Pointer;
begin
     tmp := Pointer(bit_mask[index]);

     asm
          mov eax, value
          mov ebx, tmp
          and eax, ebx
          mov tmp, eax
     end;
     if tmp = Pointer(0) then
        is_bit_set := false
     else
        is_bit_set := true;
end;

{******************************************************************************
 *   is_not_bit_set
 ******************************************************************************
 *  does:
 *   returns true if bit at given index is not set
 ******************************************************************************}
function is_bit_not_set( index: byte; value: Pointer ): Boolean; [public, alias:'IS_BIT_NOT_SET'];
var
   tmp: Pointer;
begin
     tmp := Pointer(bit_mask[index]);

     asm
          mov eax, value
          mov ebx, tmp
          and eax, ebx
          mov tmp, eax
     end;

     if tmp = Pointer(0) then
        is_bit_not_set := true
     else
        is_bit_not_set := false;
end;

{******************************************************************************
 *   high
 ******************************************************************************
 *   returns high word of DWord value
 ******************************************************************************}
function high( value: DWord ): Word; [public, alias:'HIGH_D'];
var
   tmp: word;
begin
     asm
        mov eax, value
        shr eax, 16
        mov tmp, ax
     end;
     high := tmp;
end;

{******************************************************************************
 *   high
 ******************************************************************************
 *   returns high byte of Word value
 ******************************************************************************}
function high( value: word ): byte; [public, alias:'HIGH_W'];
var
   tmp: byte;
begin
     asm
        mov ax, value
        mov tmp, ah
     end;
     high := tmp;
end;

{******************************************************************************
 *   low
 ******************************************************************************
 *   returns low word of DWord value
 ******************************************************************************}
function low( value: DWord ): Word; [public, alias:'LOW_D'];
var
   tmp: word;
begin
     asm
        mov eax, value
        mov tmp, ax
     end;
     low := tmp;
end;

{******************************************************************************
 *   low
 ******************************************************************************
 *   returns low byte of Word value
 ******************************************************************************}
function low( value: Word ): byte; [public, alias:'LOW_W'];
var
   tmp: byte;
begin
     asm
        mov ax, value
        mov tmp, al
     end;
     low := tmp;
end;

begin
end.

